// -----------------------------------------------------------------------------
// Project:	TToolbarEditor
// Module:	tberuned - Run time editor form interface
// -----------------------------------------------------------------------------
unit tberuned;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  StdCtrls, ExtCtrls,
  tbEdit;

type
  // Abstract run time editor base class
  TFormRunToolEdit = class(TForm)
    private
      FToolbarEditor: TToolbarEditor;

    protected
      procedure BeginDrag;
      procedure DoClose;
      procedure DoReset;
      procedure DoHelp;
      function GetButton: TToolbarItem; virtual; abstract;

    public
      constructor Create(AOwner: TComponent); override;

      property ToolbarEditor: TToolbarEditor read FToolbarEditor write FToolbarEditor;
  end;

  TFormDefaultToolbarEditor = class(TFormRunToolEdit)
    BevelFrame: TBevel;
    LabelCategories: TLabel;
    LabelCommands: TLabel;
    ListBoxCategories: TListBox;
    ListBoxCommands: TListBox;
    LabelHelp: TLabel;
    ButtonReset: TButton;
    ButtonClose: TButton;
    ButtonHelp: TButton;
    procedure ListBoxCategoriesClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure ListBoxCommandsMeasureItem(Control: TWinControl;
      Index: Integer; var Height: Integer);
    procedure ListBoxCommandsDrawItem(Control: TWinControl; Index: Integer;
      ItemRect: TRect; State: TOwnerDrawState);
    procedure ListBoxCommandsMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ButtonCloseClick(Sender: TObject);
    procedure ButtonResetClick(Sender: TObject);
    procedure ButtonHelpClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);

  private

  protected
    function GetButton: TToolbarItem; override;

  public

  end;

var
  FRuntimeToolbarEditor: TFormRunToolEdit;

implementation

uses
  buttons;

{$R *.DFM}

// -----------------------------------------------------------------------------
//
//			TFormRunToolEdit
//
// -----------------------------------------------------------------------------
constructor TFormRunToolEdit.Create(AOwner: TComponent);
begin
  inherited;
  { This can safely be removed }
  if (AOwner is TToolbarEditor) then
    ToolbarEditor := TToolbarEditor(AOwner);
end;

// -----------------------------------------------------------------------------
procedure TFormRunToolEdit.BeginDrag;
var
  DragButton		: TSpeedButton;
  ToolbarItem		: TToolbarItem;
begin
  ToolbarItem := GetButton;
  if (ToolbarItem = nil) then
    exit;
  if (ToolbarItem.Button = nil) then
    DragButton := ToolbarItem.CreateButton(ToolbarEditor.ToolBar)
  else
    DragButton := (ToolbarItem.Button as TSpeedButton);
  DragButton.BeginDrag(False);
end;

// -----------------------------------------------------------------------------
procedure TFormRunToolEdit.DoClose;
begin
  FRuntimeToolbarEditor := nil;
  if (Assigned(ToolbarEditor.OnAfterEdit)) then
    ToolbarEditor.OnAfterEdit(Owner);
  // Resyncronize buttons after edit
  ToolbarEditor.SynchronizeWithMenu;
end;

// -----------------------------------------------------------------------------
procedure TFormRunToolEdit.DoReset;
begin
  if (Assigned(ToolbarEditor.OnEditReset)) then
    ToolbarEditor.OnEditReset(Owner);
end;

// -----------------------------------------------------------------------------
procedure TFormRunToolEdit.DoHelp;
begin
  if (Assigned(ToolbarEditor.OnEditHelp)) then
    ToolbarEditor.OnEditHelp(Owner);
end;

// -----------------------------------------------------------------------------
//
//			TFormDefaultToolbarEditor
//
// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.FormShow(Sender: TObject);
begin
  // Only display buttons if required
  ButtonReset.Visible := (toRuntimeReset in ToolbarEditor.Options);
  ButtonHelp.Visible := (toRuntimeHelp in ToolbarEditor.Options);

  // Move Close button if Help button not visible
  if not(ButtonHelp.Visible) then
    ButtonClose.Left := ButtonHelp.Left;

  // Get categories
  ListBoxCategories.Items.Assign(ToolbarEditor.Categories);

  // Select first category
  if (ListBoxCategories.Items.Count > 0) then
    ListBoxCategories.ItemIndex := 0;

  ListBoxCategoriesClick(Sender);
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ListBoxCategoriesClick(Sender: TObject);
var
  i			: integer;
begin
  // Empty listbox
  ListBoxCommands.Items.Clear;

  if (ListBoxCategories.ItemIndex < 0) then
    exit;

  // Find all items that belongs to the selected category
  // and add them to the listbox
  for i:=0 to ToolbarEditor.ItemCount-1 do
    if (ToolbarEditor.Items[i].CategoryIndex =
      ListBoxCategories.ItemIndex) then
      ListBoxCommands.Items.AddObject(ToolbarEditor.Items[i].Description,
        ToolbarEditor.Items[i]);
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ListBoxCommandsMeasureItem(Control: TWinControl;
  Index: Integer; var Height: Integer);
begin
  height := TToolbarItem(ToolbarEditor.Items[Index]).Height + 2;
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ListBoxCommandsDrawItem(Control: TWinControl;
  Index: Integer; ItemRect: TRect; State: TOwnerDrawState);
var
  ToolbarItem		: TToolbarItem;
begin
  // Clear the drawing area
  TListBox(Control).Canvas.FillRect(ItemRect);

  // Make space between rows
  InflateRect(ItemRect, -1, -1);

  ToolbarItem := TToolbarItem(TListBox(Control).Items.Objects[Index]);

  ToolbarItem.DrawButton(TListBox(Control).Canvas, ItemRect.Left, ItemRect.Top);

  // Draw the description text
  if (ToolbarItem.Description <> '') then
  begin
    // Make left margin
    if (ToolbarItem.Width < 30) then
      ItemRect.Left := ItemRect.left + 34
    else
      ItemRect.Left := ItemRect.left + ToolbarItem.Width + 4;
    // Center text vertically
    DrawText(TListBox(Control).Canvas.Handle, PChar(ToolbarItem.Description),
      -1, ItemRect, DT_VCENTER or DT_SINGLELINE);
  end;

end;


// -----------------------------------------------------------------------------
function TFormDefaultToolbarEditor.GetButton: TToolbarItem;
begin
  if (ListBoxCommands.ItemIndex = -1) then
    Result := nil
  else
    Result := TToolbarItem(ListBoxCommands.Items.Objects[ListBoxCommands.ItemIndex]);
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ListBoxCommandsMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) then
    BeginDrag;
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ButtonResetClick(Sender: TObject);
begin
  DoReset;
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ButtonHelpClick(Sender: TObject);
begin
  DoHelp;
end;

// -----------------------------------------------------------------------------
procedure TFormDefaultToolbarEditor.ButtonCloseClick(Sender: TObject);
begin
  Close;
end;

procedure TFormDefaultToolbarEditor.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  DoClose;
  Action := caFree;
end;

// -----------------------------------------------------------------------------
end.
